///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//  Copyright  NetworkDLS 2002, All rights reserved
//
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF 
// ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO 
// THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A 
// PARTICULAR PURPOSE.
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef _NEW_DB_CPP
#define _NEW_DB_CPP
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include <Windows.H>
#include <WindowsX.H>
#include <ShellAPI.H>
#include <Stdio.H>
#include <Stdlib.H>
#include <SQL.H>
#include <SQLExt.H>

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include "../Resources/Resource.H"

#include "../../SharedClasses/SQLClass/cSQL.H"
#include "../../SharedClasses/SQLClass/cRecordSet.H"
#include "../../SharedClasses/CGetPKs/CGetPKs.H"
#include "../../SharedClasses/CMemPool/CMemPool.H"
#include "../../SharedSource/SQLTemplate.H"
#include "../CSockSrvr/CSockSrvr.H"

#include "../../SharedClasses/CStatusDlg/CStatusDlg.H"

#include "../../SharedSource/Debug.H"
#include "../../SharedSource/NSWFL.H"
#include "../../SharedSource/Common.H"
#include "../../SharedSource/CRC32.H"

#include "Init.H"
#include "Routines.H"
#include "Command.H"
#include "Console.H"
#include "SQLImport.H"
#include "NewDB.H"

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool SetupNewClient(char *sClientDBFileName, char *sDBPrefix, char *sDBO, bool bImportStructure, 
					bool bImportData, bool bDropImportTable, CStatusDlg *lpDlg)
{
	/*
		sClientDBFileName: Contains the name of the backup file that contains the users existing
			data and structure. (Relative to the server)

		sDBName: Contains the base name of the new DB to create for the new client.

		bImportStructure: Should we import the structure of the sClientDBFileName to our new DB?

		bImportData: Should we import the data from the sClientDBFileName to our new DB?
	*/

    CSQL MySQL;
	CSQL CustSQL;
	CRecordSet rsObjs;

	char sSQL[1024];
	char sFullDBName[1024];
	char sDBName[1024];

	if(bImportData && !bImportStructure)
	{
		lpDlg->SetText("You cannot import data if you opt not to import the DB structure.");
		return false;
	}

	lpDlg->SetText("Connecting to the SQL server.");
	if(!MySQL.Connect(gsSQLCustDriver, gsSQLCustServer, gsSQLCustUserID, gsSQLCustPassword, "Master", gsDBMode))
    {
        lpDlg->SetText("Error connecting to the [Master] database.");
        return false;
    }

	if(!GetDBNameFromBackup(&MySQL, sClientDBFileName, sDBName))
	{
        lpDlg->SetText("Failed to load the database backup file information.");
        return false;
	}

	sprintf_s(sFullDBName, sizeof(sFullDBName), "SQLExch_%s_%s", sDBPrefix, sDBName);

	if(bImportStructure || bImportData)
	{
		lpDlg->SetText("Restoring the import database. (This could take some time)");
		if(!RestoreDBForImport(&MySQL, sClientDBFileName, sFullDBName))
		{
			lpDlg->SetText("Failed to restore the clients database.");
			MySQL.Disconnect();
			return false;
		}
	}

	lpDlg->SetText("Creating the new client database.");
	if(!CreateDatabaseIfNotExist(&MySQL, NULL, -1, sFullDBName))
	{
		lpDlg->SetText("Failed to create the new database.");
		MySQL.Disconnect();
		return false;
	}

	lpDlg->SetText("Connecting to the SQL server.");
	if(!CustSQL.Connect(gsSQLCustDriver, gsSQLCustServer, gsSQLCustUserID, gsSQLCustPassword, sFullDBName, gsDBMode))
    {
        lpDlg->SetText("Error connecting to the [Master] database.");
        return false;
    }

	lpDlg->SetText("Dropping any existing database structure.");
	sprintf_s(sSQL, sizeof(sSQL),
		"SELECT [Name], [xType]"
		" FROM [%s].[%s].[SysObjects]"
		" WHERE [xType] IN ('U', 'TR')",
		sFullDBName, sDBO);

	CustSQL.Execute(sSQL, &rsObjs);

	char sType[100];
	char sObj[1024];
	int iSz = 0;
	int iProgress = 0;

	CustSQL.bThrowErrors = false;
	rsObjs.bThrowErrors = false;

	iProgress = 0;
	lpDlg->SetProgressRange(0, rsObjs.RowCount);
	
	while(rsObjs.Fetch())
	{
		rsObjs.sColumnEx(1, sObj, sizeof(sObj), &iSz);
		rsObjs.sColumnEx(2, sType, sizeof(sType), &iSz);

		if(_strcmpi(sType, "U") == 0)
		{
			sprintf_s(sSQL, sizeof(sSQL), "DROP TABLE [%s]", sObj);
		}
		else if(_strcmpi(sType, "TR") == 0)
		{
			sprintf_s(sSQL, sizeof(sSQL), "DROP TRIGGER [%s]", sObj);
		}
		CustSQL.ExecuteNonQuery(sSQL);
		
		iProgress++;

		lpDlg->SetProgressPos(iProgress);
	}

	rsObjs.Close();

	CustSQL.Disconnect();

	if(bImportStructure || bImportData)
	{
		lpDlg->SetText("Importing the database structure.");
		if(!ImportDBStructure(&MySQL, sDBO, sFullDBName, bImportData, lpDlg))
		{
			lpDlg->SetText("Failed to import the database structure.");
			MySQL.Disconnect();
			return false;
		}
	}

	if(bDropImportTable)
	{
		lpDlg->SetText("Dropping the import database.");
		sprintf_s(sSQL, sizeof(sSQL), "DROP DATABASE [%s_TMP]", sFullDBName);
		MySQL.ExecuteNonQuery(sSQL);
	}

	MySQL.Disconnect();

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool ImportDBStructure(CSQL *lpMySQL, char *sDBO, char *sDBName, bool bImportData, CStatusDlg *lpDlg)
{
	char *sBaseSQL = NULL;
	char sSQL[5120];

	int iTempSz = 0;

	FILE *hSource = NULL;
	
	char *sLargeSQL = NULL;
	char sTable[1024];
	long lTableID = 0;

	char sColName[1024];
	char sColType[255];
	long lColLen = 0;
	long lColStatus = 0;
	long lColNullable = 0;

	char sSmlTmp[32];
	char sLgTmp[1024];

	int iPK = 0;
	int iRow = 0;
	int iLargeSQLSz = 1024 * 1024;

	CGetPKs cMyPKs;

	CRecordSet rsTables;
	CRecordSet rsColumns;

	sBaseSQL = "SELECT [Name], [ID]"
		" FROM [%s_TMP].[%s].[SysObjects]"
		" WHERE xType = 'U' AND [Name] <> 'dtProperties' AND LEFT([Name], 8) <> 'SQLExch_'"
		" ORDER BY [Name]";
	sprintf_s(sSQL, sizeof(sSQL), sBaseSQL, sDBName, gsDefaultDBO);

	if(!lpMySQL->Execute(sSQL, &rsTables))
	{
		lpMySQL->GetErrorMessage(sLgTmp, sizeof(sLgTmp), rsTables.hSTMT);
		
		sprintf_s(sSQL, sizeof(sSQL), "Failed to import the database structure: %s", sLgTmp);
		WriteSysLogEx(sSQL, EVENT_ERROR);

		return false;
	}

	lpDlg->SetProgressRange(0, rsTables.RowCount);
	
	sLargeSQL = (char *) gMem.Allocate(iLargeSQLSz, sizeof(char));
	Assert(!sLargeSQL, "Memory Allocation Error.");

	while(rsTables.Fetch())
	{
		rsTables.sColumnEx(1, sTable, sizeof(sTable), &iTempSz);
		lTableID = rsTables.lColumn(2);

		sBaseSQL =
			"SELECT [Columns].[Name], [Columns].[Length], [Types].[Name],"
			" (CASE WHEN [Types].[Name] IN ('Binary', 'Char', 'nChar', 'nVarChar', 'VarBinary', 'VarChar') THEN 1 ELSE 0 END) AS [TypeStatus],"
			" [Columns].[IsNullable]"
			" FROM [%s_TMP].[%s].[SysColumns] AS [Columns]"
			" INNER JOIN [%s_TMP].[%s].[SysTypes] AS [Types] ON [Columns].[xType] = [Types].[xUserType]"
			" WHERE [Columns].[ID] = %d"
			" ORDER BY [Columns].[ColOrder]";
		sprintf_s(sSQL, sizeof(sSQL), sBaseSQL, sDBName, sDBO, sDBName, sDBO, lTableID);

		if(!lpMySQL->Execute(sSQL, &rsColumns))
		{
			lpMySQL->GetErrorMessage(sLgTmp, sizeof(sLgTmp), rsColumns.hSTMT);
			sprintf_s(sSQL, sizeof(sSQL), "Failed to import the database structure: %s", sLgTmp);
			WriteSysLogEx(sSQL, EVENT_ERROR);

			rsTables.Close();
			gMem.Free(sLargeSQL);
			return false;
		}

		//------------------------------------------------------------------------------------------
		//Drop the existing table
		//------------------------------------------------------------------------------------------
		lpMySQL->bThrowErrors = false;
		sprintf_s(sSQL, sizeof(sSQL), "DROP TABLE [%s].[%s].[%s]", sDBName, sDBO, sTable);
		lpMySQL->ExecuteNonQuery(sSQL);
		lpMySQL->bThrowErrors = true;
		//------------------------------------------------------------------------------------------

		sprintf_s(sLargeSQL, iLargeSQLSz,
			"CREATE TABLE [%s].[%s].[%s] (\r\n", sDBName, sDBO, sTable);

		sprintf_s(sLgTmp, sizeof(sLgTmp), "%s_TMP", sDBName);
		cMyPKs.Get(lpMySQL, sLgTmp, sDBO, sTable);

		while(rsColumns.Fetch())
		{
			rsColumns.sColumnEx(1, sColName, sizeof(sColName), &iTempSz);
			lColLen = rsColumns.lColumn(2);
			rsColumns.sColumnEx(3, sColType, sizeof(sColType), &iTempSz);
			lColStatus = rsColumns.lColumn(4);
			lColNullable = rsColumns.lColumn(5);

			strcat_s(sLargeSQL, iLargeSQLSz, "\t[");
			strcat_s(sLargeSQL, iLargeSQLSz, sColName);
			strcat_s(sLargeSQL, iLargeSQLSz, "] [");
			strcat_s(sLargeSQL, iLargeSQLSz, sColType);
			strcat_s(sLargeSQL, iLargeSQLSz, "]");
			if(lColStatus == 1)
			{
				if(lColLen <= 0)
				{
					strcat_s(sLargeSQL, iLargeSQLSz, "(MAX)");
				}
				else{
					sprintf_s(sSmlTmp, sizeof(sSmlTmp), " (%d)", lColLen);
					strcat_s(sLargeSQL, iLargeSQLSz, sSmlTmp);
				}
			}
			
			if(lColNullable == 0)
			{
				strcat_s(sLargeSQL, iLargeSQLSz, " NOT NULL,\r\n");
			}
			else {
				strcat_s(sLargeSQL, iLargeSQLSz, " NULL,\r\n");
			}
		}

		iTempSz = strlen(sLargeSQL);
		sLargeSQL[iTempSz - 1] = '\0';
		sLargeSQL[iTempSz - 2] = '\0';
		sLargeSQL[iTempSz - 3] = '\0';
		strcat_s(sLargeSQL, iLargeSQLSz, "\r\n");

		if(cMyPKs.iPKs > 0)
		{
			strcat_s(sLargeSQL, iLargeSQLSz, "\tCONSTRAINT [");
			strcat_s(sLargeSQL, iLargeSQLSz, cMyPKs.sPKName);
			strcat_s(sLargeSQL, iLargeSQLSz, "] PRIMARY KEY CLUSTERED\r\n");
			strcat_s(sLargeSQL, iLargeSQLSz, "\t(\r\n");

			iPK = 0;
			while(iPK < cMyPKs.iPKs)
			{
				strcat_s(sLargeSQL, iLargeSQLSz, "\t\t[");
				strcat_s(sLargeSQL, iLargeSQLSz, cMyPKs.sPKs[iPK]);
				strcat_s(sLargeSQL, iLargeSQLSz, "]");

				if(iPK != (cMyPKs.iPKs - 1))
				{
					strcat_s(sLargeSQL, iLargeSQLSz, ",\r\n");
				}
				iPK++;
			}
			strcat_s(sLargeSQL, iLargeSQLSz, "\r\n\t) ON [PRIMARY]\r\n");
			
		}

		strcat_s(sLargeSQL, iLargeSQLSz, ") ON [PRIMARY]");

		lpMySQL->ExecuteNonQuery(sLargeSQL);

		if(bImportData)
		{
			//------------------------------------------------------------------------------------------
			//Import the data.
			//------------------------------------------------------------------------------------------
			sprintf_s(sSQL, sizeof(sSQL), 
				"INSERT INTO [%s].[%s].[%s]"
				" SELECT * FROM [%s_TMP].[%s].[%s]",
				sDBName, sDBO, sTable, sDBName, sDBO, sTable);
			lpMySQL->ExecuteNonQuery(sSQL);
			//------------------------------------------------------------------------------------------
		}

		cMyPKs.Free();
		rsColumns.Close();

		iRow++;

		lpDlg->SetProgressPos(iRow);
	}
	
	gMem.Free(sLargeSQL);
	rsTables.Close();

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool GetDBNameFromBackup(CSQL *lpMySQL, const char *sBackupFile, char *sOut_DBName)
{
	char sSQL[5120];

	int iColumn = 1;
	int iSz = 0;

	bool bResult = false;

	CRecordSet rsTemp;

	sprintf_s(sSQL, sizeof(sSQL), "RESTORE HEADERONLY FROM DISK = '%s'", sBackupFile);
	if(!lpMySQL->Execute(sSQL, &rsTemp))
	{
		MsgBox("The backup file specified is either invalid or does not exist.");
		return false;
	}

    char sColName[255];
	int iColNameSz = sizeof(sColName);
	
	while(rsTemp.GetColInfo(iColumn, sColName, iColNameSz, NULL, NULL, NULL, NULL, NULL))
	{
		if(_strcmpi(sColName, "DatabaseName") == 0)
		{
			if(rsTemp.Fetch())
			{
				if(rsTemp.sColumnEx(iColumn, sOut_DBName, 1024, &iSz))
				{
					bResult = true;
				}
			}
		}
		
		iColNameSz = sizeof(sColName);
		iColumn++;
	}

	rsTemp.Close();

	return bResult;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool RestoreDBForImport(CSQL *lpMySQL, char *sClientDBFileName, char *sDBName)
{
	char *sBaseSQL = NULL;
	char sSQL[5120];

	int iTempSz = 0;
	char sTemp1[1024];
	char sTemp2[1024];

	char sLogicalData[1024];
	char sLogicalLog[1024];

	CRecordSet rsTemp;

	sprintf_s(sSQL, sizeof(sSQL), "RESTORE FILELISTONLY FROM DISK = '%s'", sClientDBFileName);
	if(!lpMySQL->Execute(sSQL, &rsTemp))
	{
		MsgBox("The backup file specified is either invalid or does not exist.");
		return false;
	}

	strcpy_s(sLogicalData, sizeof(sLogicalData), "");
	strcpy_s(sLogicalLog, sizeof(sLogicalLog), "");

	while(rsTemp.Fetch())
	{
		rsTemp.sColumnEx(1, sTemp1, sizeof(sTemp1), &iTempSz);

		rsTemp.sColumnEx(3, sTemp2, sizeof(sTemp2), &iTempSz);
		if(_strcmpi(sTemp2, "D") == 0)
		{
			//Data File.
			strcpy_s(sLogicalData, sizeof(sLogicalLog), sTemp1);
		}
		else if(_strcmpi(sTemp2, "L") == 0)
		{
			//Log File.
			strcpy_s(sLogicalLog, sizeof(sLogicalLog), sTemp1);
		}
	}

	rsTemp.Close();

	if(strlen(sLogicalData) > 0 && strlen(sLogicalLog) > 0)
	{
		sBaseSQL = "RESTORE DATABASE [%s_TMP] FROM DISK = '%s'\n"
			"WITH REPLACE, RECOVERY, MOVE '%s' TO '%s\\%s_TMP_Dat.mdf',\n"
			"MOVE '%s' TO '%s\\%s_TMP_Log.ldf'\n";

		sprintf_s(sSQL, sizeof(sSQL), sBaseSQL, sDBName, sClientDBFileName,
			sLogicalData, gsSQLDataFiles, sDBName,
			sLogicalLog, gsSQLDataFiles, sDBName);
	}
	else{
		MsgBox("Failed to obtain a list of logical file names from the backup file specified.");
		return false;
	}

	return lpMySQL->ExecuteNonQuery(sSQL);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool CreateNewDB(CSQL *lpMySQL, char *sDBName)
{
	char sSQL[5120];
	SQLTEMPLATE MySQLTemp;

	memset(&MySQLTemp, 0, sizeof(MySQLTemp));

	MySQLTemp.iMask = SQLTMP_DBDIR | SQLTMP_DB;
	MySQLTemp.sDBDir = gsSQLDataFiles;
	MySQLTemp.sDB = sDBName;
	if(BuildSQLFromTemplate(gsPath, "SQLExch_Create_DB.sql", sSQL, &MySQLTemp))
	{
		return lpMySQL->ExecuteNonQuery(sSQL);
	}

	return false;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool CreateDatabaseIfNotExist(CSQL *lpMySQL, CSockSrvr *pSockSrvr, int iClient, const char *sDB)
{
	char sTemp[5120];
	CRecordSet rsExist;

	sprintf_s(sTemp, sizeof(sTemp), "SELECT Count(*) FROM [SysDatabases] WHERE [SysDatabases].[Name] = '%s'", sDB);
	lpMySQL->Execute(sTemp, &rsExist);

	if(!rsExist.Fetch())
	{
		if(pSockSrvr)
		{
			WriteLogEx(pSockSrvr->icClientID[iClient], "Failed to fetch the SQL data.", EVENT_ERROR);
		}
		return false;	
	}

	if(rsExist.lColumn(1) < 1)
	{
		sprintf_s(sTemp, sizeof(sTemp), "The client database [%s] does not exist. Creating it...", sDB);
		if(pSockSrvr)
		{
			WriteLogEx(pSockSrvr->icClientID[iClient], sTemp, EVENT_NONE);
		}

		SQLTEMPLATE MySQLTemp;

		memset(&MySQLTemp, 0, sizeof(MySQLTemp));

		MySQLTemp.iMask = SQLTMP_DBDIR | SQLTMP_DB;
		MySQLTemp.sDBDir = gsSQLDataFiles;
		MySQLTemp.sDB = (char *) sDB;

		if(BuildSQLFromTemplate(gsPath, "SQLExch_Create_DB.sql", sTemp, &MySQLTemp))
		{
			if(!lpMySQL->ExecuteNonQuery(sTemp))
			{
				if(pSockSrvr)
				{
					WriteLogEx(pSockSrvr->icClientID[iClient], "Failed to create the customer database.", EVENT_ERROR);
				}
				return false;			
			}

			sprintf_s(sTemp, sizeof(sTemp), 
				"UPDATE [Companys]"
				" SET [InitStep] = 0"
				" WHERE [CompName] = '%s'",
				CCI[iClient].sCompanyName);
				CCI[iClient].cIndexSQL.ExecuteNonQuery(sTemp);
				CCI[iClient].bRequestInit = true;
		}
		else{
			if(pSockSrvr)
			{
				WriteLogEx(pSockSrvr->icClientID[iClient], "Failed to obtain DB creation SQL.", EVENT_ERROR);
			}
			return false;			
		}
	}

	rsExist.Close();
	
	return true;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool CreateNewDBFromDBCreationScript(CSQL *lpMySQL, CSockSrvr *pSockSrvr, int iClient, char *sFileName)
{
	FILE *fSource = NULL;
	int iLen = 0;
	int iTables = 0;
	int iTable = 0;

	bool bThrowSQLErr = lpMySQL->bThrowErrors;

	bool bResult = true;

	char sTemp[1024];
	char sSQL[1024];
	char sDB[255];
	char sTable[255];
	char sPKName[255];
	char *sLargeSQL1 = NULL;
	//SQLTEMPLATE MySQLTemp;

	//memset(&MySQLTemp, 0, sizeof(MySQLTemp));

	if(fopen_s(&fSource, sFileName, "rb") != 0)
	{
		return false;
	}

	sLargeSQL1 = (char *) gMem.Allocate(sizeof(char), 1024 * 100);
	Assert(!sLargeSQL1, "Memory Allocation Error.");

	/*
	MySQLTemp.iMask = SQLTMP_DBDIR | SQLTMP_DB;
	MySQLTemp.sDBDir = gsSQLDataFiles;
	MySQLTemp.sDB = CCI[iClient].sCompanyDatabase;
	*/

	//----------------------------------------------------------------------------------------------------

	int iBytesRead = 0;

	while(true)
	{
		bool bFirstRecord = true;

		iTable = 0;

		//Read Table count.
		if(fread(&iTables, sizeof(iTables), 1, fSource) == 0)
		{
			bResult = true;
			break;
		}

		while(iTable < iTables && pSockSrvr->bcConnected[iClient] && !pSockSrvr->bcDisconnect[iClient])
		{
			//Read Database Name.
			fread(&iLen, sizeof(iLen), 1, fSource);
			fread(sDB, sizeof(char), iLen, fSource);
			sDB[iLen] = '\0';

			strcpy_s(sTemp, sizeof(sTemp), sDB);
			strcpy_s(sDB, sizeof(sDB), "SQLExch_");
			strcat_s(sDB, sizeof(sDB), CCI[iClient].sCompanyDBPrefix);
			strcat_s(sDB, sizeof(sDB), "_");
			strcat_s(sDB, sizeof(sDB), sTemp);
			
			if(bFirstRecord)
			{
				if(!CreateDatabaseIfNotExist(&CCI[iClient].cCustSQL, pSockSrvr, iClient, sDB))
				{
					bResult = false;
					break;
				}

				bFirstRecord = false;
			}

			//Read Table Name.
			fread(&iLen, sizeof(iLen), 1, fSource);
			fread(sTable, sizeof(char), iLen, fSource);
			sTable[iLen] = '\0';

			sprintf_s(sSQL, sizeof(sSQL), "Creating table: %s.", sTable);
			WriteLogEx(pSockSrvr->icClientID[iClient], sSQL, EVENT_NONE);

			//Read PK Name.
			fread(&iLen, sizeof(iLen), 1, fSource);
			fread(sPKName, sizeof(char), iLen, fSource);
			sPKName[iLen] = '\0';

			//Read SQL.
			fread(&iLen, sizeof(iLen), 1, fSource);
			fread(sLargeSQL1, sizeof(char), iLen, fSource);
			sLargeSQL1[iLen] = '\0';

			//Drop the existing table.
			lpMySQL->bThrowErrors = false;
			sprintf_s(sSQL, sizeof(sSQL), "DROP TABLE [%s].[DBO].[%s]", sDB, sTable);
			lpMySQL->ExecuteNonQuery(sSQL);
			lpMySQL->bThrowErrors = true;

			ReplStr(sLargeSQL1, "{SQLExch_TAG_DB}", sDB);

			//Recreate the table.
			if(!lpMySQL->ExecuteNonQuery(sLargeSQL1))
			{
				bResult = false;
				break;
			}

			iTable++;
		}

		if(bResult == false)
		{
			break;
		}

		if(pSockSrvr->bcConnected[iClient] == false || pSockSrvr->bcDisconnect[iClient] == true)
		{
			break;
		}

	}

	lpMySQL->bThrowErrors = bThrowSQLErr;

	fclose(fSource);
	gMem.Free(sLargeSQL1);

	if(pSockSrvr->bcConnected[iClient] && !pSockSrvr->bcDisconnect[iClient] && bResult == true)
	{
		return true;
	}

	return false;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#endif
